# 機能設計書 24-名前変更 (Rename)

## 概要

本ドキュメントは、Roslynにおける「名前変更 (Rename)」機能の設計を記述するものである。Rename機能は、シンボル(変数、メソッド、クラス等)の名前をソリューション全体で安全に変更し、参照箇所も同時に更新する機能である。

### 本機能の処理概要

Rename機能は、指定されたシンボルの名前を新しい名前に変更し、そのシンボルへのすべての参照を自動的に更新する機能である。名前変更による競合(名前の衝突、スコープの問題等)を検出し、可能な限り自動解決を行う。

**業務上の目的・背景**：ソフトウェア開発において、変数名やメソッド名をより適切な名前に変更することは、コードの可読性・保守性を向上させるために頻繁に行われる作業である。手動での名前変更は、参照漏れや名前衝突のリスクがあり、大規模なコードベースでは特に困難である。自動化されたRename機能により、これらのリスクを排除し、安全かつ効率的に名前変更を行うことができる。

**機能の利用シーン**：
1. 変数・メソッド・クラス等の名前をより適切な名前に変更する場合
2. リファクタリングの一環として識別子を統一する場合
3. ファイル名と型名を同期させる場合
4. 名前空間をフォルダ構造と同期させる場合

**主要な処理内容**：
1. 対象シンボルのすべての参照箇所を検索(FindRenameLocationsAsync)
2. 名前変更による競合の検出
3. 競合の自動解決(修飾名の追加、名前空間の追加等)
4. すべての参照箇所の更新
5. ドキュメント名・名前空間の同期(オプション)

**関連システム・外部連携**：
- OOP(Out of Process)サービスとの連携でリモート実行可能
- FindReferences機能との連携
- エディタのインラインRename UI

**権限による制御**：特になし。すべてのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | インラインRenameダイアログ | 主画面 | 新しい名前の入力とプレビュー |
| 20 | プレビューペイン | 補助画面 | 変更箇所のプレビュー表示 |

## 機能種別

コード変換処理 / リファクタリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| solution | Solution | Yes | 対象のソリューション | nullでないこと |
| symbol | ISymbol | Yes | 名前変更対象のシンボル | nullでないこと |
| newName | string | Yes | 新しい名前 | 空でない文字列であること |
| options | SymbolRenameOptions | Yes | 名前変更オプション | - |
| cancellationToken | CancellationToken | Yes | キャンセルトークン | - |

### SymbolRenameOptions

| オプション名 | 型 | デフォルト | 説明 |
|-------------|-----|-----------|------|
| RenameOverloads | bool | false | オーバーロードも同時にリネーム |
| RenameInStrings | bool | false | 文字列リテラル内も検索・置換 |
| RenameInComments | bool | false | コメント内も検索・置換 |
| RenameFile | bool | false | 対応するファイル名も変更 |

### 入力データソース

- ソリューション全体のシンボル参照情報
- セマンティックモデル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Solution | Solution | 名前変更後の新しいソリューション |
| ConflictResolution | ConflictResolution | 競合解決結果(内部API) |
| IsSuccessful | bool | 成功したかどうか |
| ErrorMessage | string | エラーメッセージ(失敗時) |

### 出力先

- 名前変更後の新しいソリューションとして返却
- 競合解決の結果と関連位置情報

## 処理フロー

### 処理シーケンス

```
1. 引数バリデーション
   └─ solution, symbol, newNameがnull/空でないことを確認
2. OOP実行試行
   └─ RemoteHostClient経由でリモートサービス呼び出しを試みる
3. 参照箇所の検索
   └─ SymbolicRenameLocations.FindLocationsInCurrentProcessAsyncで全参照を検索
4. 競合解決
   └─ ConflictResolver.ResolveSymbolicLocationConflictsInCurrentProcessAsyncで競合を解決
5. 結果返却
   └─ ConflictResolutionを返却(成功時はNewSolutionを含む)
```

### フローチャート

```mermaid
flowchart TD
    A[開始: RenameSymbolAsync] --> B{引数チェック}
    B -->|失敗| C[ArgumentException]
    B -->|成功| D[Logger.LogBlock開始]
    D --> E{OOPクライアント利用可能?}
    E -->|Yes| F[IRemoteRenamerService.RenameSymbolAsync]
    E -->|No| G[RenameSymbolInCurrentProcessAsync]
    F --> H{リモート成功?}
    H -->|Yes| I[結果をRehydrate]
    H -->|No| G
    G --> J[FindLocationsInCurrentProcessAsync]
    J --> K[ResolveSymbolicLocationConflictsInCurrentProcessAsync]
    K --> L{競合解決成功?}
    L -->|Yes| M[NewSolutionを返却]
    L -->|No| N[ArgumentException]
    I --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 空文字列禁止 | newNameは空でない文字列である必要がある | 常時 |
| BR-02 | オーバーロード連動 | RenameOverloads=trueの場合、オーバーロードも同時にリネーム | オプション指定時 |
| BR-03 | 文字列内検索 | RenameInStrings=trueの場合、文字列リテラル内も検索・置換 | オプション指定時 |
| BR-04 | コメント内検索 | RenameInComments=trueの場合、コメント内も検索・置換 | オプション指定時 |
| BR-05 | 競合解決失敗時 | 競合が解決できない場合はArgumentExceptionをスロー | 公開API使用時 |

### 計算ロジック

競合解決アルゴリズム:
1. 名前変更後に同名の識別子が存在するかチェック
2. 存在する場合、完全修飾名や名前空間プレフィックスで区別可能かチェック
3. 解決可能な場合は自動的に修飾を追加
4. 解決不可能な場合はエラーとして報告

## データベース操作仕様

該当なし（インメモリ操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentNullException | solution, symbolがnull | 適切な引数を渡す |
| - | ArgumentException | newNameが空文字列 | 有効な名前を指定 |
| - | ArgumentException | 競合解決失敗 | エラーメッセージを確認し対処 |

### リトライ仕様

なし（OOP失敗時はローカルプロセスにフォールバック）

## トランザクション仕様

該当なし（ソリューション変更はイミュータブル）

## パフォーマンス要件

- Logger.LogBlockでFunctionId.Renamer_RenameSymbolAsyncをロギング
- OOP(Out of Process)実行でUIスレッドをブロックしない
- OOP失敗時はローカルプロセスにフォールバック

## セキュリティ考慮事項

- 特になし

## 備考

- SymbolicRenameLocationsは重量級の型で、すべての参照情報を保持
- LightweightRenameLocationsはOOP通信用の軽量版
- ConflictResolverで競合の検出と自動解決を行う

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Rename機能で使用される主要なデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | RenameOptions.cs | `src/Workspaces/Core/Portable/Rename/RenameOptions.cs` | オプション定義 |
| 1-2 | ConflictResolution.cs | `src/Workspaces/Core/Portable/Rename/ConflictResolution.cs` | 競合解決結果 |
| 1-3 | RenameLocation.cs | `src/Workspaces/Core/Portable/Rename/RenameLocation.cs` | 参照位置情報 |

**読解のコツ**: SymbolRenameOptionsの各フラグがどのように処理に影響するかを把握する。

#### Step 2: エントリーポイントを理解する

処理の起点となるRenamer.csを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Renamer.cs | `src/Workspaces/Core/Portable/Rename/Renamer.cs` | 公開API |

**主要処理フロー**:
1. **41-65行目**: RenameSymbolAsync(公開API) - 引数検証とConflictResolutionの取得
2. **142-182行目**: RenameSymbolAsync(内部) - OOP試行とローカルフォールバック
3. **184-203行目**: RenameSymbolInCurrentProcessAsync - ローカル実行

#### Step 3: 参照箇所の検索を理解する

シンボルのすべての参照を検索するロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SymbolicRenameLocations.cs | `src/Workspaces/Core/Portable/Rename/SymbolicRenameLocations.cs` | 参照検索 |
| 3-2 | LightweightRenameLocations.cs | `src/Workspaces/Core/Portable/Rename/LightweightRenameLocations.cs` | 軽量版 |

#### Step 4: 競合解決を理解する

名前変更による競合の検出と解決ロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ConflictResolver.cs | `src/Workspaces/Core/Portable/Rename/ConflictEngine/ConflictResolver.cs` | 競合解決 |
| 4-2 | ConflictResolver.Session.cs | `src/Workspaces/Core/Portable/Rename/ConflictEngine/ConflictResolver.Session.cs` | セッション管理 |

### プログラム呼び出し階層図

```
Renamer.RenameSymbolAsync (public)
    │
    ├─ 引数バリデーション
    │
    └─ Renamer.RenameSymbolAsync (internal)
           │
           ├─ SerializableSymbolAndProjectId.TryCreate
           │
           ├─ RemoteHostClient.TryGetClientAsync
           │      └─ IRemoteRenamerService.RenameSymbolAsync (OOP)
           │
           └─ RenameSymbolInCurrentProcessAsync
                  │
                  ├─ SymbolicRenameLocations.FindLocationsInCurrentProcessAsync
                  │      └─ 全参照箇所の検索
                  │
                  └─ ConflictResolver.ResolveSymbolicLocationConflictsInCurrentProcessAsync
                         └─ 競合の検出と解決
```

### データフロー図

```
[入力]               [処理]                          [出力]

ISymbol ───▶ FindLocationsInCurrentProcessAsync ───▶ SymbolicRenameLocations
                      │
                      ▼
RenameLocations ───▶ ResolveSymbolicLocationConflicts ───▶ ConflictResolution
                      │
                      ▼
ConflictResolution ───▶ (IsSuccessful判定) ───▶ Solution or Exception
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Renamer.cs | `src/Workspaces/Core/Portable/Rename/Renamer.cs` | ソース | 公開API |
| RenameOptions.cs | `src/Workspaces/Core/Portable/Rename/RenameOptions.cs` | ソース | オプション定義 |
| ConflictResolution.cs | `src/Workspaces/Core/Portable/Rename/ConflictResolution.cs` | ソース | 競合解決結果 |
| SymbolicRenameLocations.cs | `src/Workspaces/Core/Portable/Rename/SymbolicRenameLocations.cs` | ソース | 参照検索 |
| LightweightRenameLocations.cs | `src/Workspaces/Core/Portable/Rename/LightweightRenameLocations.cs` | ソース | 軽量版参照検索 |
| ConflictResolver.cs | `src/Workspaces/Core/Portable/Rename/ConflictEngine/ConflictResolver.cs` | ソース | 競合解決 |
| ConflictResolver.Session.cs | `src/Workspaces/Core/Portable/Rename/ConflictEngine/ConflictResolver.Session.cs` | ソース | セッション管理 |
| RenameLocation.cs | `src/Workspaces/Core/Portable/Rename/RenameLocation.cs` | ソース | 参照位置 |
| IRemoteRenamerService.cs | `src/Workspaces/Core/Portable/Rename/IRemoteRenamerService.cs` | ソース | OOPサービスインターフェース |
| SymbolicRenameInfo.cs | `src/Features/Core/Portable/Rename/SymbolicRenameInfo.cs` | ソース | UI向け情報 |
